//-----------------------------------------------------------------------------
// VST Plug-Ins SDK
// User interface framework for VST plugins : Standard Control Objects
//
// Version 2.0 Beta 4       Date : 20/03/00
//
// First version            : Wolfgang Kundrus         06.97
// Added new objects        : Michael Schmidt          08.97
// Added new objects        : Yvan Grabit              01.98
// Added BeOS version       : Georges-Edouard Berenger 05.99
//
// (c)1999-2000 Steinberg Soft+Hardware GmbH, All Rights Reserved
//-----------------------------------------------------------------------------

#ifndef __vstcontrols__
#define __vstcontrols__

#ifndef __vstgui__
#include "vstgui.h"
#endif

//------------------
// defines
//------------------
#ifndef kPI
#define kPI    3.14159265358979323846
#endif

#ifndef k2PI
#define k2PI   6.28318530717958647692
#endif

#ifndef kPI_2
#define kPI_2  1.57079632679489661923f
#endif

#ifndef kPI_4
#define kPI_4  0.78539816339744830962
#endif

#ifndef kE
#define kE     2.7182818284590452354
#endif

#ifndef kLN2
#define kLN2   0.69314718055994530942
#endif

#ifndef kSQRT2
#define kSQRT2 1.41421356237309504880
#endif

//------------------
// CControlEnum type
//------------------
enum CControlEnum
{
	kHorizontal = 1 << 0,
	kVertical   = 1 << 1,
	kShadowText = 1 << 2,
	kLeft       = 1 << 3,
	kRight      = 1 << 4,
	kTop        = 1 << 5,
	kBottom     = 1 << 6,
	k3DIn       = 1 << 7,
	k3DOut      = 1 << 8,
	kPopupStyle = 1 << 9,
	kCheckStyle = 1 << 10,
	kNoTextStyle = 1 << 11,
	kNoDrawStyle = 1 << 12
};

//-----------------------------------------------------------------------------
//-----------------------------------------------------------------------------
class CControlListener
{
public:
	virtual void valueChanged (CDrawContext *pContext, CControl *pControl) = 0;
};

//-----------------------------------------------------------------------------
//-----------------------------------------------------------------------------
class CControl : public CView
{
public:
	CControl (CRect &size, CControlListener *listener = 0, int tag = 0, 
						CBitmap *pBackground = 0);
	virtual ~CControl ();

	virtual void  draw (CDrawContext *pContext) = 0;
	virtual void  doIdleStuff () { if (pParent) pParent->doIdleStuff (); }

	virtual void  setValue (float val) { value = val; }
	virtual float getValue () { return value; };

	virtual void  setMin (float val) { vmin = val; }
	virtual float getMin () { return vmin; }
	virtual void  setMax (float val) { vmax = val; }
	virtual float getMax () { return vmax; }

	virtual void  setOldValue (float val) { oldValue = val; }
	virtual	float getOldValue (void) { return oldValue; }
	virtual void  setDefaultValue (float val) { defaultValue = val; }
	virtual	float getDefaultValue (void) { return defaultValue; }

	inline  int   getTag () { return tag; }

	virtual bool  isDirty ();
	virtual void  setDirty (const bool val = true);

	virtual void     setBackground (CBitmap* pBackground);
	virtual CBitmap *getBackground () { return pBackground; }

protected:
	CControlListener *listener;
	long  tag;
	float oldValue;
	float defaultValue;
	float value;
	float vmin;
	float vmax;

	CBitmap *pBackground;
};


//-----------------------------------------------------------------------------
//-----------------------------------------------------------------------------
class COnOffButton : public CControl
{
public:
	COnOffButton (CRect &size, CControlListener *listener, int tag,
                  CBitmap *background);
	virtual ~COnOffButton ();

	virtual void draw (CDrawContext*);
	virtual void mouse (CDrawContext *pContext, CPoint &where);
};


//-----------------------------------------------------------------------------
//-----------------------------------------------------------------------------
class CParamDisplay : public CControl
{
public:
	CParamDisplay (CRect &size, CBitmap *background = 0, int style = 0);
	virtual ~CParamDisplay ();
	
	virtual void setFont (CFont fontID);

	virtual void setFontColor (CColor color);
	CColor getFontColor () { return fontColor; }

	virtual void setBackColor (CColor color);
	CColor getBackColor () { return backColor; }

	virtual void setFrameColor (CColor color);
	CColor getFrameColor () { return frameColor; }

	virtual void setShadowColor (CColor color);
	CColor getShadowColor () { return shadowColor; }

	virtual void setHoriAlign (CHoriTxtAlign hAlign);
	virtual void setBackOffset (CPoint &offset);
	virtual void setStringConvert (void (*stringConvert) (float value, char *string));
	virtual void setStringConvert (void (*stringConvert2) (float value, char *string,
															void *userDta), void *userData);

	virtual void setStyle (int val);
	virtual int getStyle () { return style; }

	virtual void draw (CDrawContext *pContext);

protected:
	void drawText (CDrawContext *pContext, char *string, CBitmap *newBack = 0);

	CHoriTxtAlign horiTxtAlign;
	int     style;

	CFont   fontID;
	CColor  fontColor;
	CColor  backColor;
	CColor  frameColor;
	CColor  shadowColor;
	CPoint  offset;
	void    *userData;

private:
	void (*stringConvert) (float value, char *string);
	void (*stringConvert2) (float value, char *string, void *userData);
};


//-----------------------------------------------------------------------------
//-----------------------------------------------------------------------------
class CTextEdit : public CParamDisplay
{
public:
	CTextEdit (CRect &size, CControlListener *listener, int tag, const char *txt = 0,
               CBitmap *background = 0, int style = 0);
	~CTextEdit ();

	virtual void setText (char *txt);
	virtual void getText (char *txt);

	virtual	void draw (CDrawContext *pContext);
	virtual	void mouse (CDrawContext *pContext, CPoint &where);

	virtual void setTextEditConvert (void (*editConvert) (char *input, char *string));
	virtual void setTextEditConvert (void (*editConvert2) (char *input, char *string,
										void *userDta), void *userData);

	virtual	void takeFocus ();
	virtual	void looseFocus ();

protected:
	void *platformControl;
	void *platformFont;
	char text[256];

private:
	void (*editConvert) (char *input, char *string);
	void (*editConvert2) (char *input, char *string, void *userData);
};


//-----------------------------------------------------------------------------
//-----------------------------------------------------------------------------
#define MAX_ENTRY 1024
 class COptionMenu : public CParamDisplay
{
public:
	COptionMenu (CRect &size, CControlListener *listener, int tag,
                 CBitmap *background = 0, CBitmap *bgWhenClick = 0,
                 int style = 0);
	~COptionMenu ();

	virtual	bool addEntry (char *txt, int index = -1);
	virtual	int  getCurrent (char *txt = 0, bool countSeparator = true);
	virtual	bool setCurrent (int index, bool countSeparator = true);
	virtual	bool getEntry (int index, char *txt);
	virtual	bool removeEntry (int index);
	virtual	bool removeAllEntry ();
	virtual int  getNbEntries () { return nbEntries; }
	virtual int  getIndex (char *txt);

	virtual	void draw (CDrawContext *pContext);
	virtual	void mouse (CDrawContext *pContext, CPoint &where);

	virtual	void takeFocus ();
	virtual	void looseFocus ();

#if MOTIF
	void setCurrentSelected (void *itemSelected);
#endif

protected:
	void    *platformControl;
	char    *entry[MAX_ENTRY];

#if MOTIF
	void    *itemWidget[MAX_ENTRY];
#endif

	int      nbEntries;
	int      currentIndex;
	CBitmap *bgWhenClick;
};


//-----------------------------------------------------------------------------
//-----------------------------------------------------------------------------
class CKnob : public CControl
{
public:
	CKnob (CRect &size, CControlListener *listener, int tag, 
           CBitmap *background,
           CBitmap *handle, CPoint &offset);
	virtual ~CKnob ();

	virtual void draw (CDrawContext*);
	virtual	void mouse (CDrawContext *pContext, CPoint &where);

	virtual void  valueToPoint (CPoint &point);
	virtual float valueFromPoint (CPoint &point);

	virtual void setInsetValue (int val) { inset = val; }

protected:
	CBitmap *pHandle;
	int      inset;
	CPoint   offset;
};

//-----------------------------------------------------------------------------
//-----------------------------------------------------------------------------
class CAnimKnob : public CKnob
{
public:
	CAnimKnob (CRect &size, CControlListener *listener, int tag, 
               int subPixmaps,        // number of subPixmaps
               int heightOfOneImage,  // pixel
               CBitmap *handle, CPoint &offset);
	virtual ~CAnimKnob ();

	virtual void draw (CDrawContext*);

protected:
	int subPixmaps;		// number of subPixmaps
	int heightOfOneImage;
};


//-----------------------------------------------------------------------------
//-----------------------------------------------------------------------------
class CVerticalSwitch : public CControl
{
public:
	CVerticalSwitch (CRect &size, CControlListener *listener, int tag, 
                     int subPixmaps,         // number of subPixmaps
                     int heightOfOneImage,   // pixel
                     int iMaxPositions,
                     CBitmap *background, CPoint &offset);
	virtual ~CVerticalSwitch ();

	virtual void draw (CDrawContext*);
	virtual void mouse (CDrawContext *pContext, CPoint &where);

protected:
	CPoint   offset;
	int      subPixmaps;            // number of subPixmaps
	int      heightOfOneImage;
	int      iMaxPositions;
};


//-----------------------------------------------------------------------------
//-----------------------------------------------------------------------------
class CHorizontalSwitch : public CControl
{
public:
	CHorizontalSwitch (CRect &size, CControlListener *listener, int tag, 
                       int subPixmaps,        // number of subPixmaps
                       int heightOfOneImage,  // pixel
                       int iMaxPositions,
                       CBitmap *background,
                       CPoint &offset);
	virtual	~CHorizontalSwitch ();

	virtual void draw (CDrawContext*);
	virtual void mouse (CDrawContext *pContext, CPoint &where);

protected:
	CPoint   offset;
	int      subPixmaps;        // number of subPixmaps
	int      heightOfOneImage;
	int      iMaxPositions;
};


//-----------------------------------------------------------------------------
//-----------------------------------------------------------------------------
class CRockerSwitch : public CControl
{
public:
	CRockerSwitch (CRect &size, CControlListener *listener, int tag, 
                   int heightOfOneImage,  // pixel
                   CBitmap *background, CPoint &offset);
	virtual ~CRockerSwitch ();

	virtual void draw (CDrawContext*);
	virtual void mouse (CDrawContext *pContext, CPoint &where);

protected:
	CPoint   offset;
	int      heightOfOneImage;
};


//-----------------------------------------------------------------------------
//-----------------------------------------------------------------------------
class CMovieBitmap : public CControl
{
public:
	CMovieBitmap (CRect &size, CControlListener *listener, int tag, 
                  int subPixmaps,        // number of subPixmaps
                  int heightOfOneImage,  // pixel
                  CBitmap *background, CPoint &offset);
	virtual	~CMovieBitmap ();

	virtual void draw (CDrawContext*);

protected:
	CPoint   offset;
	int      subPixmaps;         // number of subPixmaps
	int      heightOfOneImage;
};


//-----------------------------------------------------------------------------
//-----------------------------------------------------------------------------
class CMovieButton : public CControl
{
public:
	CMovieButton (CRect &size, CControlListener *listener, int tag, 
                  int heightOfOneImage,  // pixel
                  CBitmap *background, CPoint &offset);
	virtual ~CMovieButton ();	

	virtual void draw (CDrawContext*);
	virtual void mouse (CDrawContext *pContext, CPoint &where);

protected:
	CPoint   offset;
	int      heightOfOneImage;
	float    buttonState;
};


//-----------------------------------------------------------------------------
//-----------------------------------------------------------------------------
// displays bitmaps within a (child-) window
class CAutoAnimation : public CControl
{
public:
	CAutoAnimation (CRect &size, CControlListener *listener, int tag, 
                    int subPixmaps,        // number of subPixmaps...
                    int heightOfOneImage,  // pixel
                    CBitmap *background, CPoint &offset);
	virtual ~CAutoAnimation ();

	virtual void draw (CDrawContext*);
	virtual void mouse (CDrawContext *pContext, CPoint &where);

	virtual void openWindow (void);
	virtual void closeWindow (void);

	virtual void nextPixmap (void);
	virtual void previousPixmap (void);

	bool    isWindowOpened () { return windowOpened; }

protected:
	CPoint   offset;

	int      subPixmaps;
	int      heightOfOneImage;

	bool     windowOpened;
	int      totalHeightOfBitmap;
};


//-----------------------------------------------------------------------------
//-----------------------------------------------------------------------------
// Vertical Slider
class CVerticalSlider : public CControl
{
public:
	CVerticalSlider (CRect &size, CControlListener *listener, int tag, 
                     int     iMinYPos,    // min Y position in pixel
                     int     iMaxYPos,    // max Y position in pixel
                     CBitmap *handle,     // bitmap slider
                     CBitmap *background, // bitmap background
                     CPoint  &offset,
                     int     style = kBottom); // style (kBottom, kTop))

	virtual ~CVerticalSlider ();
  
	virtual void draw (CDrawContext*);
	virtual void mouse (CDrawContext *pContext, CPoint &where);

	virtual void setDrawTransparentHandle (bool val) { bDrawTransparentEnabled = val; }
	virtual void setFreeClick (bool val) { bFreeClick = val; }
	virtual bool getFreeClick () { return bFreeClick; }
	virtual void setOffsetHandle (CPoint &val) { offsetHandle = val; }

	virtual void     setHandle (CBitmap* pHandle);
	virtual CBitmap *getHandle () { return pHandle; }

protected:
 	CBitmap *pHandle;

	int      widthOfSlider; // size of the handle-slider
	int      heightOfSlider;

	CPoint   offset; 
	CPoint   offsetHandle;

	int      iMinYPos;     // min Y position in pixel
	int      iMaxYPos;     // max Y position in pixel
	int      style;

	int      actualYPos;
	bool     bDrawTransparentEnabled;
	bool     bFreeClick;

	int      minTmp;
	int      maxTmp;
	int      widthControl;
	int      heightControl;
};

//-----------------------------------------------------------------------------
//-----------------------------------------------------------------------------
// Horizontal Slider
class CHorizontalSlider : public CControl
{
public:
	CHorizontalSlider (CRect &size, CControlListener *listener, int tag, 
                       int     iMinXPos,    // min X position in pixel
                       int     iMaxXPos,    // max X position in pixel
                       CBitmap *handle,     // bitmap slider
                       CBitmap *background, // bitmap background	
                       CPoint  &offset,
                       int     style = kRight); // style (kRight, kLeft));
  
	virtual ~CHorizontalSlider ();
  
	virtual void draw (CDrawContext*);
	virtual void mouse (CDrawContext *pContext, CPoint &where);

	virtual void setDrawTransparentHandle (bool val) { bDrawTransparentEnabled = val; }
	virtual void setFreeClick (bool val) { bFreeClick = val; }
	virtual bool getFreeClick () { return bFreeClick; }
	virtual void setOffsetHandle (CPoint &val) { offsetHandle = val; }

	virtual void     setHandle (CBitmap* pHandle);
	virtual CBitmap *getHandle () { return pHandle; }

protected:
 	CBitmap *pHandle;

	int      widthOfSlider;      // size of the handle-slider
	int      heightOfSlider;

	CPoint   offset; 
 	CPoint   offsetHandle;
 
	int      iMinXPos;     // min X position in pixel
	int      iMaxXPos;     // max X position in pixel
	int      style;

	int      actualXPos;
	bool     bDrawTransparentEnabled;
	bool     bFreeClick;

	int      minTmp;
	int      maxTmp;
	int      widthControl;
	int      heightControl;
};


//-----------------------------------------------------------------------------
//-----------------------------------------------------------------------------
// special display with custom numbers (0...9)
class CSpecialDigit : public CControl
{
public:
	CSpecialDigit (CRect &size, CControlListener *listener, int tag, // tag identifier
                   long     dwPos,     // actual value
                   int      iNumbers,  // amount of numbers (max 7)
                   int      *xpos,     // array of all XPOS
                   int      *ypos,     // array of all YPOS
                   int      width,     // width of ONE number
                   int      height,    // height of ONE number
                   CBitmap  *background);  // bitmap numbers
	virtual ~CSpecialDigit ();
	
	virtual void  draw (CDrawContext*);

	virtual float getNormValue (void);

protected:
	int      iNumbers;   // amount of numbers
	int      xpos[7];    // array of all XPOS, max 7 possible
	int      ypos[7];    // array of all YPOS, max 7 possible
	int      width;      // width  of ONE number
	int      height;     // height of ONE number
};


//-----------------------------------------------------------------------------
//-----------------------------------------------------------------------------
class CKickButton : public CControl
{
public:
	CKickButton (CRect &size, CControlListener *listener, int tag, 
                 int heightOfOneImage,  // pixel
                 CBitmap *background, CPoint &offset);
	virtual ~CKickButton ();	

	virtual void draw (CDrawContext*);
	virtual void mouse (CDrawContext *pContext, CPoint &where);

protected:
	CPoint   offset;
	int      heightOfOneImage;
};


//-----------------------------------------------------------------------------
//-----------------------------------------------------------------------------
class CSplashScreen : public CControl
{
public:
	CSplashScreen (CRect &size, CControlListener *listener, int tag, 
                   CBitmap *background,
                   CRect &toDisplay, 
                   CPoint &offset);
	virtual ~CSplashScreen ();	
  
	virtual void draw (CDrawContext*);
	virtual void mouse (CDrawContext *pContext, CPoint &where);
  
protected:
	CRect    toDisplay;
	CRect    keepSize;
	CPoint   offset;
};


//-----------------------------------------------------------------------------
//-----------------------------------------------------------------------------
class CVuMeter : public CControl
{
public:
	CVuMeter (CRect& size, CBitmap *onBitmap, CBitmap *offBitmap,
              int nbLed, const int style = kVertical);
	virtual ~CVuMeter ();	
  
	virtual void setDecreaseStepValue (float value) { decreaseValue = value; }

	virtual void draw (CDrawContext *pContext);

protected:
	CBitmap *onBitmap;
	CBitmap *offBitmap;
	int      nbLed;
	int      style;
	float    decreaseValue;

	CRect    rectOn;
	CRect    rectOff;
};

#endif

